<?php

declare(strict_types=1);

namespace NeuronAI\Chat\Messages;

use NeuronAI\Chat\Messages\ContentBlocks\AudioContent;
use NeuronAI\Chat\Messages\ContentBlocks\ContentBlockInterface;
use NeuronAI\Chat\Messages\ContentBlocks\ImageContent;
use NeuronAI\Chat\Messages\ContentBlocks\ReasoningContent;
use NeuronAI\Chat\Messages\ContentBlocks\TextContent;
use NeuronAI\Chat\Enums\MessageRole;
use NeuronAI\StaticConstructor;
use JsonSerializable;

use function array_map;
use function array_merge;
use function is_string;

/**
 * @method static static make(MessageRole $role, string|ContentBlockInterface|ContentBlockInterface[]|null $content = null)
 */
class Message implements JsonSerializable
{
    use StaticConstructor;
    use HasMetadata;

    protected ?Usage $usage = null;

    /**
     * @var ContentBlockInterface[]
     */
    protected array $contents = [];

    /**
     * @param string|ContentBlockInterface|ContentBlockInterface[]|null $content
     */
    public function __construct(
        protected MessageRole $role,
        string|ContentBlockInterface|array|null $content = null
    ) {
        if ($content !== null) {
            $this->setContents($content);
        }
    }

    public function getRole(): string
    {
        return $this->role->value;
    }

    public function setRole(MessageRole|string $role): Message
    {
        if (!$role instanceof MessageRole) {
            $role = MessageRole::from($role);
        }

        $this->role = $role;
        return $this;
    }

    /**
     * @return ContentBlockInterface[]
     */
    public function getContentBlocks(): array
    {
        return $this->contents;
    }

    /**
     * @param string|ContentBlockInterface|ContentBlockInterface[] $content
     */
    public function setContents(string|ContentBlockInterface|array $content): Message
    {
        if (is_string($content)) {
            $this->contents = [new TextContent($content)];
        } elseif ($content instanceof ContentBlockInterface) {
            $this->contents = [$content];
        } else {
            // Assume it's an array
            foreach ($content as $block) {
                $this->addContent($block);
            }
        }

        return $this;
    }

    public function addContent(ContentBlockInterface $block): Message
    {
        $this->contents[] = $block;

        return $this;
    }

    /**
     * Get the text content of the message.
     */
    public function getContent(): ?string
    {
        $text = '';
        foreach ($this->getContentBlocks() as $index => $block) {
            if ($block instanceof TextContent) {
                $text .= ($index > 0 ? " " : '').$block->content;
            }
            if ($block instanceof ReasoningContent) {
                $text .= ($index > 0 ? "\n\n" : '').$block->content."\n\n";
            }
        }

        return $text === '' || $text === '0' ? null : $text;
    }

    public function getAudio(): ?AudioContent
    {
        foreach ($this->getContentBlocks() as $block) {
            if ($block instanceof AudioContent) {
                return $block;
            }
        }

        return null;
    }

    public function getImage(): ?ImageContent
    {
        foreach ($this->getContentBlocks() as $block) {
            if ($block instanceof ImageContent) {
                return $block;
            }
        }

        return null;
    }

    public function getUsage(): ?Usage
    {
        return $this->usage;
    }

    public function setUsage(Usage $usage): static
    {
        $this->usage = $usage;
        return $this;
    }

    /**
     * @return array<string, mixed>
     */
    public function jsonSerialize(): array
    {
        $data = [
            'role' => $this->getRole(),
            'content' => array_map(fn (ContentBlockInterface $block): array => $block->toArray(), $this->contents)
        ];

        if ($this->getUsage() instanceof Usage) {
            $data['usage'] = $this->getUsage()->jsonSerialize();
        }

        return array_merge($this->meta, $data);
    }
}
